import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:wordpress_app/blocs/config_bloc.dart';
import 'package:wordpress_app/cards/card1.dart';
import 'package:wordpress_app/cards/card4.dart';
import 'package:wordpress_app/config/config.dart';
import 'package:wordpress_app/services/app_service.dart';
import 'package:wordpress_app/services/wordpress_service.dart';
import 'package:wordpress_app/utils/cached_image.dart';
import 'package:wordpress_app/utils/empty_image.dart';
import 'package:wordpress_app/utils/loading_card.dart';
import 'package:wordpress_app/widgets/inline_ads.dart';
import 'package:wordpress_app/widgets/loading_indicator_widget.dart';
import 'package:wordpress_app/widgets/scroll_to_top_button.dart';
import '../models/article.dart';
import 'package:easy_localization/easy_localization.dart';

class AuthorArticles extends ConsumerStatefulWidget {
  final Article article;
  const AuthorArticles({super.key, required this.article});

  @override
  ConsumerState<AuthorArticles> createState() => _AuthorArticlesState();
}

class _AuthorArticlesState extends ConsumerState<AuthorArticles> {
  final List<Article> _articles = [];
  ScrollController? _controller;
  int page = 1;
  bool? _loading;
  bool? _hasData;
  var scaffoldKey = GlobalKey<ScaffoldState>();
  final int _postAmount = 10;

  @override
  void initState() {
    _controller =
        ScrollController(initialScrollOffset: 0.0, keepScrollOffset: true);
    _controller!.addListener(_scrollListener);
    _hasData = true;
    _fetchData();
    super.initState();
  }

  Future _fetchData() async {
    await WordPressService()
        .fetchPostsByAuthor(page, widget.article.authorId!, _postAmount)
        .then((value) {
      setState(() {
        if (value.isNotEmpty) {
          _articles.addAll(value);
          _loading = false;
          if (_articles.isEmpty) {
            _hasData = false;
          }
        }
      });
    });
  }

  @override
  void dispose() {
    super.dispose();
    _controller!.dispose();
  }

  Future<void> _scrollListener() async {
    var isEnd = AppService.isScrollNearEnd(_controller!);
    if (isEnd && _articles.isNotEmpty) {
      setState(() {
        page += 1;
        _loading = true;
      });
      await _fetchData().then((value) {
        setState(() {
          _loading = false;
        });
      });
    }
  }

  Future _onRefresh() async {
    setState(() {
      _loading = null;
      _hasData = true;
      _articles.clear();
      page = 1;
    });
    _fetchData();
  }

  @override
  Widget build(BuildContext context) {
    final configs = ref.read(configBlocProvider).configs!;
    return Scaffold(
      body: RefreshIndicator(
        child: Stack(
          children: [
            CustomScrollView(
              controller: _controller,
              slivers: <Widget>[
                SliverAppBar.large(
                  automaticallyImplyLeading: false,
                  pinned: true,
                  leading: const BackButton(
                    color: Colors.white,
                  ),
                  backgroundColor: Theme.of(context).primaryColor,
                  elevation: 0.5,
                  flexibleSpace: _flexibleSpaceBar(context),
                ),
                _hasData == false
                    ? SliverFillRemaining(
                        child: Column(
                        children: [
                          SizedBox(
                            height: MediaQuery.of(context).size.height * 0.20,
                          ),
                          EmptyPageWithImage(
                              image: Config.noContentImage,
                              title: 'no-contents'.tr()),
                        ],
                      ))
                    : _articles.isEmpty && _hasData == true
                        ? SliverPadding(
                            padding: const EdgeInsets.fromLTRB(15, 15, 15, 15),
                            sliver: SliverList.separated(
                              itemCount: 6,
                              itemBuilder: (context, index) {
                                return const LoadingCard(height: 250);
                              },
                              separatorBuilder: (context, index) {
                                return const Divider(height: 40);
                              },
                            ),
                          )
                        : SliverPadding(
                            padding: const EdgeInsets.all(20),
                            sliver: SliverList.separated(
                              itemCount: _articles.length,
                              itemBuilder: (context, index) {
                                if ((index + 1) % configs.postIntervalCount ==
                                    0) {
                                  return Column(
                                    children: [
                                      Card4(
                                          article: _articles[index],
                                          heroTag:
                                              'author_${_articles[index].id}'),
                                      const InlineAds(isSliver: true),
                                    ],
                                  );
                                } else {
                                  return Card1(
                                      article: _articles[index],
                                      heroTag:
                                          'author1_${_articles[index].id}');
                                }
                              },
                              separatorBuilder: (context, index) {
                                return const Divider(height: 40);
                              },
                            ),
                          ),
                if (_articles.isNotEmpty && _loading == true)
                  const SliverToBoxAdapter(
                    child: Padding(
                      padding: EdgeInsets.all(30.0),
                      child: LoadingIndicatorWidget(),
                    ),
                  ),
              ],
            ),
            Positioned(
              bottom: 50,
              right: 16,
              child: ScrollToTopButton(controller: _controller!),
            ),
          ],
        ),
        onRefresh: () async => _onRefresh(),
      ),
    );
  }

  FlexibleSpaceBar _flexibleSpaceBar(BuildContext context) {
    return FlexibleSpaceBar(
      centerTitle: true,
      background: Container(
        color: Theme.of(context).primaryColor,
        width: double.infinity,
        child: Center(
          child: ClipOval(
              child: SizedBox(
                  height: 50,
                  child: CustomCacheImage(
                      imageUrl: widget.article.avatar, radius: 0))),
        ),
      ),
      title: Text(widget.article.author.toString(),
          style: const TextStyle(color: Colors.white)),
      titlePadding: const EdgeInsets.all(20),
    );
  }
}
